#pragma once

#include <vector>
#include "../Cache.h"
#include "../BitStreamFactory.h"
#include "../FileStream.h"
#include "../AVIMux_GUI/AudioSource_AC3.h"

using namespace System;
using namespace NUnit::Framework;

namespace AMG { namespace AC3 
{
	AC3SOURCE* GetAc3AudioSourceFromStream(STREAM* stream)
	{
		AC3SOURCE* result = new AC3SOURCE(stream);
		Assert::AreEqual((int)STREAM_OK, (int)result->Open(stream));
		return result;
	}

	AC3SOURCE* GetAc3AudioSourceFromFile(std::wstring fileName)
	{
		CFileStream* stream = new CFileStream();
		int openResult = stream->Open(fileName.c_str(), StreamMode::Read);
		Assert::AreEqual((int)STREAM_OK, openResult);

		return GetAc3AudioSourceFromStream(stream);
	}

	AC3SOURCE* GetAc3AudioSourceFromBuffer(const unsigned char* data, unsigned int size)
	{
		try
		{
			CMemoryStream* stream = CMemoryStream::Create(reinterpret_cast<const char*>(data), size);
			return GetAc3AudioSourceFromStream(stream);
		}
		catch (Object^)
		{
			return NULL;
		}
	}

	void CloseAc3AudioSource(AC3SOURCE* ac3)
	{
		ac3->GetSource()->Close();
		delete ac3->GetSource();
		ac3->Close();
		delete ac3;
	}

	unsigned char testdata_48khz_448kbps_1792bpf[] = {
		0x0B, 0x77, 0xD9, 0xA8, 0x1E, 0x40, 0xE1, 0xDF, 0x44, 0xC0, 0x3F, 0xE0, 0xFF, 0x59, 0x7B, 0xC0,
		0x62, 0x10, 0x10, 0x7F, 0x0F, 0x0F, 0xE1, 0xE1, 0xBC, 0x04, 0x76, 0x57, 0x44, 0x1F, 0xBE, 0xAF,
		0x2A, 0xFB, 0xFC, 0xB7, 0x3C, 0xAD, 0x72, 0xCF, 0x8B, 0x3E, 0x2C, 0xF9, 0xF3, 0xE2, 0x2B, 0xCB,
		0x3C, 0xE0, 0xE9, 0xF3, 0xA7, 0xCE, 0x8B, 0x3C, 0xDF, 0x4D, 0xF3, 0x65, 0x4D, 0x38, 0x3E, 0x79,
		0x71, 0xF3, 0x64, 0x4F, 0x6B, 0xAA, 0x7C, 0xD9, 0xED, 0x71, 0xCA, 0x9F, 0x34, 0xE0, 0xFB, 0xDF,
		0x8F, 0xDC, 0xC3, 0x72, 0xFB, 0x3A, 0x5A, 0x6E, 0x94, 0xD2, 0xA6, 0xE9, 0x32, 0xA4, 0x2F, 0xD0,
		0xBB, 0x73, 0x4B, 0x5A, 0x9A, 0x4F, 0xBF, 0xF9, 0xC4, 0xB3, 0xA5, 0x4D, 0x8B, 0x3E, 0x54, 0xD8,
		0xAD, 0x77, 0xCF, 0x9F, 0x3E, 0x29, 0x71, 0xF3, 0xA7, 0xB5, 0xDF, 0x3E, 0x79, 0xC1, 0xF3, 0xE7,
		0xCF, 0x8B, 0x3E, 0x7C, 0xF9, 0xE5, 0xC7, 0x4F, 0x6B, 0xBE, 0x7C, 0xF8, 0x73, 0xE7, 0xCF, 0x9E,
		0x70, 0x54, 0xD3, 0x83, 0xEF, 0xBB, 0xFB, 0x79, 0xC1, 0xED, 0x77, 0xB5, 0xDE, 0xD3, 0x79, 0xA1,
		0xF3, 0xE7, 0xCF, 0x9E, 0xD7, 0x7C, 0xF9, 0xF3, 0xE5, 0x4F, 0x9F, 0x3E, 0x2C, 0xD9, 0xF3, 0xE2,
		0xB0, 0xDF, 0x36, 0x7C, 0xF3, 0x83, 0xE7, 0xCF, 0x9E, 0x3F, 0xAC, 0x97, 0x83, 0xE7, 0xCF, 0x9E,
		0xD7, 0x7C, 0xAB, 0xCE, 0xF8, 0xED, 0x70, 0x2C, 0xF8, 0xB3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF8,
		0xB3, 0xE7, 0x9A, 0x1F, 0x3D, 0xAE, 0xF9, 0xD3, 0x67, 0xCF, 0x9F, 0x3E, 0x54, 0xF8, 0xB3, 0xE5,
		0x4F, 0x9F, 0x3D, 0xA6, 0xF9, 0xF3, 0xE7, 0x4E, 0x93, 0x2A, 0x54, 0xF9, 0xF3, 0xE2, 0xB5, 0xDF,
		0x3E, 0xFA, 0x4F, 0xBC, 0xAF, 0xBD, 0x12, 0x24, 0x48, 0x91, 0x22, 0x44, 0x80, 0x25, 0x0E, 0xC4,
		0x96, 0x87, 0x17, 0x09, 0xBE, 0x6B, 0x78, 0xD6, 0xB5, 0xDF, 0x3E, 0x03, 0xE6, 0xDB, 0x6D, 0xB6,
		0x07, 0x80, 0xC3, 0xE7, 0xC0, 0x6B, 0x5B, 0x5D, 0x35, 0xAC, 0x9B, 0x99, 0x6F, 0x2E, 0xDB, 0xB6,
		0x6E, 0xEE, 0xB7, 0x6E, 0xA7, 0x6E, 0x9B, 0x5E, 0x17, 0xEE, 0xC4, 0x80, 0x00, 0x41, 0x80, 0x2C,
		0x73, 0xC8, 0x53, 0x8B, 0x84, 0x26, 0x04, 0x19, 0xD2, 0x49, 0x38, 0x51, 0x44, 0x08, 0x60, 0x18,
		0x7C, 0x7C, 0x58, 0x04, 0x70, 0x7C, 0x6C, 0x50, 0x05, 0xC7, 0x03, 0x07, 0x03, 0x17, 0x07, 0x1F,
		0x1F, 0x09, 0x00, 0x0C, 0x78, 0x58, 0x50, 0x30, 0xE0, 0x81, 0x12, 0x19, 0xC1, 0x00, 0x0A, 0x28,
		0x90, 0x02, 0x14, 0xE7, 0xC8, 0x7C, 0x5C, 0x4A, 0x6A, 0xDE, 0xD0, 0xE7, 0x03, 0xA2, 0x82, 0xAE,
		0x7D, 0x80, 0x87, 0xC2, 0x01, 0xEE, 0xD7, 0x7B, 0x53, 0xA4, 0x94, 0x26, 0x58, 0x84, 0xDD, 0xA1,
		0xF4, 0x07, 0x66, 0xFB, 0x59, 0xF9, 0x51, 0x82, 0xCF, 0xCF, 0xFA, 0x19, 0x6B, 0xB5, 0xD6, 0x24,
		0x6D, 0x71, 0x90, 0x3F, 0xFE, 0x19, 0x11, 0x62, 0xDA, 0xD6, 0xBC, 0x6B, 0x5B, 0xE0, 0x3E, 0x43,
		0x6D, 0xB6, 0xDB, 0x6D, 0xBE, 0x00, 0x1C, 0xBE, 0x6B, 0x6B, 0x5F, 0x35, 0xAD, 0xBA, 0x93, 0xBB,
		0xE3, 0x5B, 0x5A, 0xD6, 0xB6, 0xB7, 0xCD, 0xDF, 0x01, 0x81, 0xB6, 0xDB, 0x6D, 0xB7, 0xCF, 0x9A,
		0xD6, 0xB6, 0xED, 0x9C, 0x5A, 0xD9, 0x71, 0x8D, 0xD1, 0x44, 0x42, 0x6F, 0x9A, 0xDA, 0xE5, 0xAD,
		0x6B, 0xE7, 0xCD, 0xB6, 0xDB, 0x6D, 0x0D, 0xBE, 0x7C, 0x00, 0x3E, 0x6B, 0x5B, 0x5A, 0xDA, 0xD6,
		0xA0, 0x29, 0x23, 0xB7, 0x05, 0x03, 0x07, 0xCD, 0xE0, 0x01, 0x08, 0x08, 0x3F, 0x87, 0x87, 0xF0,
		0xF0, 0xFE, 0x1E, 0x08, 0x00, 0xB0, 0x70, 0xC8, 0x53, 0x7C, 0x99, 0xF3, 0xDB, 0x50, 0xA3, 0xBD,
		0xAE, 0xE9, 0xD1, 0x5A, 0x89, 0xFA, 0xBE, 0x02, 0x41, 0xEC, 0xC9, 0x28, 0x69, 0x70, 0x9B, 0x46,
		0xB7, 0x8D, 0x6B, 0x5D, 0xF3, 0xE0, 0x3E, 0x6D, 0xB6, 0xDB, 0x60, 0x78, 0x00, 0x3E, 0x7E, 0x06,
		0xB5, 0xB5, 0xAF, 0x9A, 0xCB, 0x35, 0x4B, 0x43, 0x47, 0x32, 0xE6, 0xB3, 0x55, 0x6B, 0x2D, 0xD5,
		0xAE, 0xCB, 0x37, 0x7D, 0x1F, 0x47, 0x19, 0xE3, 0x02, 0x04, 0xC9, 0x58, 0xA1, 0x43, 0x8D, 0x3E,
		0x00, 0x78, 0x06, 0x9E, 0x20, 0xC6, 0x8E, 0x08, 0x24, 0x00, 0x32, 0x14, 0x04, 0x10, 0x2E, 0x32,
		0x08, 0x03, 0x63, 0x82, 0xC0, 0x09, 0x8C, 0x04, 0x00, 0x05, 0x0F, 0x80, 0x10, 0x36, 0x0C, 0x24,
		0x08, 0x01, 0xCC, 0x82, 0x28, 0x91, 0x40, 0x4A, 0x3C, 0xF0, 0x33, 0x1A, 0x20, 0x2F, 0x86, 0xC6,
		0xD9, 0xE2, 0x0E, 0xF6, 0xA6, 0xC4, 0xEB, 0x82, 0x95, 0xC6, 0x7E, 0x2A, 0x9E, 0x31, 0x08, 0xF0,
		0xEF, 0x2E, 0x18, 0x20, 0x63, 0xCD, 0x1F, 0x82, 0x75, 0x72, 0xD8, 0xBF, 0x3D, 0xF8, 0x76, 0x37,
		0xE8, 0x5D, 0x75, 0x30, 0x37, 0x10, 0xF6, 0x7F, 0x0B, 0x47, 0x16, 0xCE, 0xAA, 0x03, 0x14, 0x88,
		0xB1, 0x6B, 0x6B, 0x5E, 0x35, 0xAD, 0xF0, 0x1F, 0x01, 0xB6, 0xDB, 0x6D, 0xB6, 0xDF, 0x01, 0x00,
		0x9F, 0x34, 0x35, 0xB5, 0xEC, 0xDB, 0x5E, 0x49, 0x5E, 0xF9, 0xAD, 0xAD, 0x6B, 0x43, 0x43, 0xE7,
		0x8F, 0x80, 0xC0, 0xDB, 0x6D, 0xB6, 0xDB, 0xE7, 0xCD, 0x6B, 0x59, 0x06, 0xB5, 0xAD, 0x69, 0xF8,
		0xF6, 0xB8, 0xA2, 0x21, 0x37, 0xCD, 0x6D, 0x78, 0xD6, 0xB5, 0xF3, 0xE6, 0xDB, 0x6D, 0xB6, 0x06,
		0xDF, 0x3E, 0x00, 0x1F, 0x35, 0xAD, 0xAD, 0x6D, 0x6C, 0x16, 0xA4, 0xBC, 0xB2, 0xCE, 0xAD, 0x83,
		0xE0, 0x01, 0x01, 0xBD, 0xE9, 0x58, 0x54, 0x97, 0x33, 0xF5, 0x4E, 0x69, 0xCC, 0x7F, 0x5A, 0x14,
		0x44, 0x94, 0xD2, 0xBE, 0x7E, 0x96, 0xB3, 0xEA, 0xCF, 0x9F, 0xA9, 0xB0, 0x31, 0xF5, 0x27, 0xAF,
		0xF3, 0xA5, 0xAE, 0xA6, 0x15, 0xB4, 0xB5, 0xD2, 0x53, 0x7E, 0xF6, 0xB3, 0xF6, 0xB4, 0xA1, 0x2B,
		0x41, 0x81, 0x19, 0xB6, 0x94, 0x78, 0x34, 0xB0, 0x4D, 0xF3, 0x5E, 0x06, 0xB5, 0xB6, 0xF9, 0xF0,
		0x1F, 0x36, 0xDB, 0x6D, 0xB0, 0x05, 0x06, 0x1F, 0x3E, 0x03, 0x5A, 0xDA, 0xA6, 0x50, 0x83, 0xAC,
		0xA7, 0x64, 0xA3, 0xBB, 0x6E, 0xDB, 0xAA, 0xA5, 0x8A, 0xC3, 0xDA, 0x23, 0x94, 0xBD, 0xA0, 0xED,
		0xF1, 0xCD, 0xB8, 0x22, 0x05, 0xB1, 0x21, 0x02, 0x24, 0x61, 0x72, 0x05, 0x98, 0x1C, 0xF0, 0x0B,
		0x15, 0x18, 0x6C, 0x08, 0x83, 0x02, 0xD0, 0x10, 0xDC, 0x18, 0x70, 0xF0, 0x40, 0x00, 0xC1, 0x01,
		0x23, 0x45, 0x00, 0x34, 0x7C, 0x40, 0xB0, 0xF1, 0xF0, 0x00, 0x30, 0xA1, 0x47, 0xC2, 0x17, 0x3C,
		0x59, 0x32, 0x87, 0x1A, 0x3C, 0xBA, 0x6F, 0xAC, 0x6A, 0xBF, 0xC3, 0x80, 0x39, 0x43, 0x44, 0x58,
		0x05, 0xA6, 0x31, 0x52, 0xBB, 0xCA, 0x3D, 0x50, 0x22, 0xBC, 0x93, 0x25, 0x5F, 0x18, 0x4A, 0x79,
		0x74, 0xED, 0x36, 0xBE, 0x8B, 0xE9, 0x22, 0x05, 0xC1, 0x0A, 0x15, 0x81, 0x22, 0x81, 0x01, 0x74,
		0xAD, 0x10, 0x39, 0x4C, 0x6A, 0x5E, 0x0F, 0xF4, 0x96, 0x39, 0x11, 0x78, 0xD6, 0xD7, 0xCD, 0x78,
		0xD6, 0xB7, 0xC0, 0x03, 0x6D, 0xB6, 0xDB, 0x6D, 0xBE, 0x03, 0xE0, 0x38, 0x6B, 0x6B, 0xE6, 0xB5,
		0x8D, 0xBE, 0x12, 0xBD, 0xF3, 0x5B, 0x5A, 0xD6, 0xB6, 0xBE, 0x61, 0x5F, 0x21, 0x81, 0xB6, 0xDB,
		0x6D, 0xB7, 0xCF, 0x9F, 0x35, 0xAD, 0x63, 0x5C, 0xF3, 0xD9, 0x9D, 0x64, 0x9A, 0x42, 0x48, 0x00,
		0xDA, 0xDA, 0xF1, 0xAD, 0x6B, 0xE7, 0xCF, 0x9B, 0x6D, 0xB6, 0xD8, 0x1B, 0x7C, 0x07, 0xC8, 0x6B,
		0x5B, 0x5A, 0xDA, 0xD6, 0x8A, 0xC1, 0x8C, 0x23, 0x04, 0x6B, 0x07, 0xC0, 0x00, 0x00, 0x08, 0x93,
		0xB6, 0xA2, 0xE1, 0xA5, 0x82, 0x6F, 0x9A, 0xDE, 0x35, 0xAD, 0x77, 0xCF, 0x80, 0xF9, 0xB6, 0xDB,
		0x6D, 0xA1, 0xE0, 0x30, 0xF9, 0xF2, 0x1A, 0xD0, 0xD6, 0xB4, 0x6D, 0x48, 0xA5, 0x1B, 0x72, 0x26,
		0xBD, 0xB0, 0xDB, 0xB5, 0x00, 0x3A, 0x6E, 0xE9, 0x20, 0xA8, 0x04, 0xB0, 0x8E, 0xC8, 0x0C, 0x43,
		0xA0, 0xA1, 0x8C, 0x06, 0x50, 0x23, 0x8F, 0x06, 0x04, 0x40, 0x44, 0x94, 0x18, 0xD1, 0x00, 0x01,
		0x45, 0x17, 0x1E, 0x8F, 0x00, 0xC3, 0x60, 0xC3, 0x84, 0x0D, 0x8D, 0x1B, 0x02, 0x02, 0x38, 0x80,
		0x01, 0xA1, 0xC6, 0x47, 0x88, 0x85, 0x0F, 0x81, 0x09, 0x24, 0x78, 0x19, 0x66, 0x4C, 0x03, 0x28,
		0xA3, 0xC1, 0x46, 0x24, 0x68, 0x11, 0x5F, 0x13, 0x1C, 0x7B, 0xB2, 0x06, 0x70, 0x04, 0xF3, 0x8B,
		0x09, 0xB8, 0x9E, 0x86, 0x0D, 0xC5, 0x3A, 0xB5, 0x74, 0x6E, 0xDF, 0x78, 0x92, 0xDA, 0x82, 0x11,
		0xD0, 0x60, 0x26, 0x10, 0x60, 0xDA, 0x9A, 0x54, 0x42, 0x2C, 0xDD, 0xA8, 0x08, 0xE9, 0x42, 0xAF,
		0xB7, 0xBC, 0xB8, 0x7D, 0x30, 0xF2, 0x08, 0x93, 0x16, 0xB6, 0xBE, 0x6D, 0x86, 0xB5, 0xBE, 0x00,
		0x1B, 0x6D, 0xB6, 0xDB, 0x6D, 0xF2, 0x1F, 0x01, 0xCB, 0x43, 0x5E, 0x35, 0xAD, 0x6E, 0x00, 0x95,
		0xFF, 0x9B, 0x5A, 0xD6, 0xB5, 0xB5, 0xF3, 0x78, 0xF8, 0x0F, 0x0D, 0xB6, 0xDB, 0x6D, 0xBE, 0x7C,
		0xF9, 0xAD, 0x6B, 0x5A, 0xD7, 0x9D, 0xAC, 0x97, 0x28, 0xCD, 0x12, 0x40, 0x26, 0xD6, 0xDB, 0xAD,
		0x6B, 0x5F, 0x3E, 0x7C, 0xDB, 0x6D, 0xB6, 0xC0, 0xDB, 0xE0, 0x3E, 0x03, 0x5A, 0xDA, 0xD6, 0xC6,
		0x34, 0xFB, 0x99, 0x12, 0xD8, 0x20, 0xE8, 0x3E, 0x6F, 0x01, 0x68, 0x40, 0x41, 0xFC, 0x3C, 0x3F,
		0x87, 0x86, 0xF0, 0x29, 0x00, 0x00, 0x10, 0xBB, 0x71, 0x44, 0x03, 0x0B, 0x04, 0xC5, 0x36, 0xBC,
		0x6B, 0x5A, 0xEF, 0x9F, 0x01, 0xF3, 0x6D, 0xB6, 0xDB, 0x03, 0x74, 0x21, 0xF3, 0xE0, 0x35, 0xAD,
		0xAD, 0x78, 0xE4, 0xD5, 0xB6, 0xC2, 0xE0, 0xBD, 0x52, 0xE5, 0xD6, 0xF2, 0xDD, 0x18, 0xCD, 0xD4,
		0x53, 0x59, 0xBA, 0x0B, 0x48, 0x32, 0x71, 0x41, 0x78, 0x97, 0x16, 0x7C, 0x63, 0x01, 0x87, 0x3A,
		0x09, 0x81, 0x40, 0x0C, 0xD8, 0xB3, 0x44, 0x84, 0x06, 0x2C, 0x04, 0x15, 0x89, 0xC2, 0x80, 0x83,
		0x04, 0x09, 0x12, 0x36, 0x3A, 0x20, 0x11, 0x82, 0xC0, 0x01, 0x8E, 0x8D, 0x1F, 0x17, 0x0B, 0x04,
		0x3C, 0x6C, 0x69, 0x63, 0x42, 0x1F, 0x10, 0x00, 0x47, 0x40, 0xAE, 0x13, 0x92, 0x38, 0x94, 0xF5,
		0x59, 0x11, 0x96, 0x77, 0xD8, 0x9C, 0x3F, 0x4E, 0x31, 0xA4, 0x33, 0xB7, 0xE6, 0x63, 0xBD, 0x51,
		0x00, 0xBA, 0xE0, 0x63, 0x89, 0x56, 0xFD, 0xA6, 0x3D, 0xC6, 0x08, 0x05, 0xD5, 0xD2, 0x21, 0xF3,
		0xF8, 0x29, 0xF9, 0x10, 0x3B, 0x62, 0x6F, 0x76, 0x0F, 0xF7, 0x35, 0x1F, 0x5F, 0x64, 0x91, 0x27,
		0x8D, 0x6D, 0x18, 0xD7, 0x8D, 0x6B, 0x7C, 0x80, 0x36, 0xDB, 0x6D, 0xB6, 0xDB, 0xE0, 0x3E, 0x03,
		0x86, 0xB6, 0x8C, 0x6B, 0xAC, 0xDC, 0x19, 0x1C, 0x1E, 0x35, 0xB5, 0xAD, 0x6B, 0x6B, 0xE6, 0xF1,
		0xF2, 0x1A, 0x1B, 0x6D, 0xB6, 0xDB, 0x7C, 0xF9, 0xA3, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x58, 0x8E,
		0x5D, 0x90, 0x24, 0x80, 0x4D, 0xAD, 0xAF, 0xDA, 0xD6, 0xBE, 0x7C, 0xF9, 0xB6, 0xDB, 0x6D, 0x81,
		0xB7, 0xC8, 0x7C, 0x06, 0xB4, 0x35, 0x96, 0x61, 0x1A, 0x1C, 0x5F, 0xEE, 0x41, 0xE4, 0xB0, 0x7C,
		0xC2, 0x02, 0xB0, 0x80, 0x83, 0x78, 0x14, 0x7F, 0x0F, 0x0F, 0xE1, 0xE0, 0x00, 0x00, 0x85, 0x80,
		0x25, 0xB3, 0xCC, 0x85, 0xA9, 0x99, 0xE2, 0x94, 0x70, 0xFC, 0x41, 0xDD, 0x57, 0x67, 0x26, 0x8C,
		0xB9, 0x25, 0xCC, 0x4E, 0x72, 0x10, 0x4B, 0x75, 0x42, 0x43, 0x0A, 0x04, 0xDA, 0x36, 0xBC, 0x6B,
		0x5B, 0x6F, 0x9F, 0x21, 0xF3, 0x6D, 0xB6, 0xDB, 0x03, 0xC0, 0x01, 0xF3, 0x40, 0x35, 0xAD, 0xAD,
		0x7D, 0x66, 0xB5, 0xAA, 0x9B, 0x48, 0xD9, 0x70, 0xE0, 0x5A, 0xB6, 0xCB, 0x04, 0xCD, 0xD6, 0x57,
		0x71, 0xD0, 0x8E, 0xC8, 0x9A, 0xE9, 0x09, 0x13, 0x5C, 0x04, 0x30, 0x81, 0x0E, 0x1F, 0x04, 0x60,
		0x71, 0xCD, 0x18, 0x21, 0xF3, 0x06, 0xC7, 0x06, 0x32, 0x37, 0x00, 0x0F, 0x81, 0x82, 0x86, 0x1B,
		0x04, 0x0A, 0x30, 0x32, 0x30, 0x61, 0x80, 0x87, 0x00, 0x02, 0x08, 0x1D, 0x08, 0x05, 0x01, 0x10,
		0x28, 0x40, 0xC0, 0xC1, 0x18, 0x0C, 0x61, 0x87, 0xCD, 0xB2, 0xE8, 0x58, 0x3B, 0x84, 0x40, 0xC1,
		0x64, 0xB0, 0xD8, 0xAD, 0x1C, 0x28, 0x2B, 0xF5, 0x82, 0xE6, 0x14, 0xA8, 0x5A, 0xA8, 0xA0, 0x16,
		0xCB, 0xFF, 0xFE, 0x6C, 0x40, 0xE8, 0x00, 0xAC, 0x0A, 0x7D, 0x97, 0xF9, 0x08, 0xD4, 0x75, 0x30,
		0xAB, 0x18, 0x44, 0xC2, 0x01, 0x25, 0x74, 0xE7, 0x0B, 0xF2, 0x78, 0x3E, 0x65, 0x8F, 0x27, 0xED,
		0x6D, 0x7C, 0xD7, 0x8D, 0x68, 0x7C, 0x00, 0x36, 0xDB, 0x6D, 0xB6, 0xDB, 0xE0, 0x3E, 0x03, 0x86,
		0xB6, 0xBE, 0x69, 0xE1, 0x0C, 0x31, 0x1C, 0x3F, 0x35, 0xB5, 0xAD, 0x6B, 0x6B, 0xE6, 0xF1, 0xF0,
		0x1A, 0x1B, 0x6D, 0xB6, 0xDB, 0x7C, 0xF9, 0xF3, 0x5A, 0xD6, 0x37, 0x2D, 0x6A, 0x57, 0xEE, 0x69,
		0x84, 0x24, 0x78, 0x0D, 0xB5, 0xAF, 0x1A, 0xD6, 0xBE, 0x7C, 0xF9, 0xB6, 0xDB, 0x6D, 0x81, 0xB7,
		0xC0, 0x7C, 0x06, 0xB5, 0xB5, 0xAD, 0xAD, 0x69, 0x0F, 0xF4, 0x8A, 0x00, 0x73, 0xB0, 0xCF, 0xFF
	};

	unsigned char testdata_48khz_384kbps_1536bpf[] = {
		0x0B, 0x77, 0x29, 0x80, 0x1C, 0x40, 0xE1, 0xDC, 0x5D, 0xCB, 0x80, 0x7E, 0x31, 0xFD, 0xA9, 0x68,
		0x40, 0x41, 0xE0, 0x41, 0x08, 0x08, 0x3C, 0x08, 0x21, 0x01, 0x07, 0x81, 0x04, 0x20, 0x20, 0x36,
		0x20, 0x84, 0x04, 0x06, 0xC4, 0x0A, 0xAB, 0xC7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF,
		0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E,
		0xFF, 0x9D, 0x5F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF,
		0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E,
		0xFF, 0xE7, 0x57, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3,
		0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF,
		0xBF, 0xF9, 0xD5, 0xF3, 0xE7, 0xCF, 0x9E, 0xD7, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C,
		0xF9, 0xF3, 0xE7, 0xCE, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE4, 0x4F, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3,
		0xEF, 0xFE, 0x6D, 0x7C, 0xF9, 0x53, 0xE7, 0xCF, 0x8A, 0x70, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F,
		0x3E, 0x7C, 0xE9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C,
		0xEB, 0xFF, 0x9C, 0x5F, 0x3E, 0x2C, 0xF9, 0xF3, 0xE5, 0x35, 0xDF, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7,
		0xCF, 0x9F, 0x36, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F, 0x3E, 0x7C, 0xF9, 0xF3, 0xE7, 0xCF, 0x9F,
		0x36, 0xFF, 0xE7, 0x5C, 0xA7, 0xC3, 0x12, 0x24, 0x48, 0x91, 0x22, 0x44, 0x80, 0x00, 0x00, 0x00,
		0x00, 0xDF, 0x36, 0xDB, 0x6D, 0xB6, 0xDB, 0x6D, 0xBC, 0x78, 0xF1, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
		0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDE, 0x3C, 0x78, 0xDB, 0x6D, 0xBE, 0x7C, 0xD6,
		0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0x00, 0x00,
		0x00, 0x03, 0x6D, 0xB6, 0xDB, 0x6D, 0xB6, 0xDB, 0x78, 0xF1, 0xE3, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
		0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBC, 0x78, 0xF1, 0xB6, 0xDB, 0x7C, 0xF9, 0xAD,
		0x6B, 0x5A, 0xD6, 0xB4, 0x00, 0x00, 0x00, 0x06, 0xDB, 0x6D, 0xB6, 0xDB, 0x6D, 0xB0, 0x78, 0xF1,
		0xE3, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBC, 0x78,
		0xF1, 0xB6, 0xC1, 0xBE, 0x7C, 0xD6, 0xB5, 0xAD, 0xAD, 0x6B, 0x40, 0x00, 0x00, 0x00, 0x68, 0x6D,
		0xB6, 0xD8, 0x36, 0xDB, 0x6D, 0xBC, 0x78, 0x00, 0x0E, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
		0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xF1, 0xE0, 0x1E, 0x36, 0xDB, 0x6F, 0x9F, 0x35, 0xAD, 0x6B,
		0x5A, 0xD6, 0x80, 0x00, 0x00, 0x00, 0xDB, 0x6D, 0xB6, 0x06, 0xDB, 0x6D, 0xB7, 0x8F, 0x01, 0x07,
		0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x78, 0xF0, 0x07,
		0x8D, 0xB6, 0xDB, 0xE7, 0xCD, 0x6B, 0x5A, 0xD6, 0xB5, 0xB7, 0xFE, 0x00, 0x00, 0x06, 0xF9, 0x83,
		0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1B, 0xE6, 0xDB, 0x6D, 0xB6, 0xDB, 0x6D, 0xB7, 0x8F,
		0x1E, 0x3B, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xC7,
		0x8F, 0x1B, 0x6D, 0xB7, 0xCF, 0x9A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A,
		0xD6, 0xB5, 0xAD, 0x6B, 0x40, 0x00, 0x00, 0x00, 0x6D, 0xB6, 0xDB, 0x6D, 0xB6, 0xDB, 0x6F, 0x1E,
		0x3C, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x8F,
		0x1E, 0x36, 0xDB, 0x6F, 0x9F, 0x35, 0xAD, 0x6B, 0x5A, 0xD6, 0x80, 0x00, 0x00, 0x00, 0xDB, 0x6D,
		0xB6, 0xDB, 0x6D, 0xB6, 0xEF, 0x1E, 0x3C, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
		0x77, 0x77, 0x77, 0x77, 0x77, 0x8F, 0x1E, 0x36, 0xDB, 0xB7, 0xCF, 0x9A, 0xD6, 0xB5, 0xAD, 0xAD,
		0x68, 0x00, 0x00, 0x00, 0x0C, 0x0D, 0xB6, 0xDB, 0x76, 0xDB, 0x6D, 0xB7, 0x8F, 0x00, 0x41, 0xDD,
		0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDE, 0x3C, 0x03, 0xC6,
		0xDB, 0x6D, 0xF3, 0xE6, 0xB5, 0xAD, 0x6B, 0x5A, 0xC0, 0x00, 0x00, 0x00, 0x1B, 0x6D, 0xB6, 0xC0,
		0xDB, 0x6D, 0xB6, 0xF1, 0xE4, 0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
		0xEE, 0xEE, 0xEE, 0xEF, 0x1E, 0x20, 0xF1, 0xB6, 0xDB, 0x7C, 0xF9, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5,
		0xFF, 0xC0, 0x00, 0x00, 0xDF, 0x30, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x7C, 0xDB,
		0x6D, 0xB6, 0xDB, 0x6D, 0xB6, 0xF1, 0xE3, 0xC7, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
		0x77, 0x77, 0x77, 0x77, 0x77, 0x78, 0xF1, 0xE3, 0x6D, 0xB6, 0xF9, 0xF3, 0x5A, 0xD6, 0xB5, 0xAD,
		0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x68, 0x00, 0x00, 0x00, 0x0D, 0xB6,
		0xDB, 0x6D, 0xB6, 0xDB, 0x6D, 0xE3, 0xC7, 0x8E, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
		0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xF1, 0xE3, 0xC6, 0xDB, 0x6D, 0xF3, 0xE6, 0xB5, 0xAD, 0x6B, 0x5A,
		0xD0, 0x00, 0x08, 0x00, 0x1B, 0x6D, 0xB6, 0xDB, 0x6D, 0xB6, 0xDD, 0xE3, 0xC7, 0x8E, 0xEE, 0xEE,
		0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xF1, 0xE3, 0xC6, 0xDB, 0x76,
		0xF9, 0xF3, 0x5A, 0xD6, 0x91, 0xB5, 0xAD, 0x00, 0x00, 0x00, 0x01, 0x81, 0xB6, 0xDB, 0x6E, 0xDB,
		0x6D, 0xB6, 0xF1, 0xE2, 0x00, 0x3B, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
		0xBB, 0xBB, 0xBB, 0xC7, 0x90, 0x6E, 0xDB, 0x6D, 0xBE, 0x7C, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0x00,
		0x00, 0x00, 0x03, 0x6D, 0xB6, 0xDA, 0x1B, 0x6D, 0xB6, 0xDE, 0x3C, 0x00, 0x1D, 0xDD, 0xDD, 0xDD,
		0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xE3, 0xC0, 0x1E, 0x36, 0xDB, 0x6F,
		0x9F, 0x35, 0xAD, 0x6B, 0x5A, 0xD6, 0xBF, 0xF8, 0x00, 0x00, 0x1B, 0xE6, 0x0F, 0x80, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x6F, 0x9B, 0x6D, 0xB6, 0xDB, 0x6D, 0xB6, 0xDE, 0x3C, 0x78, 0xEE, 0xEE,
		0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEF, 0x1E, 0x3C, 0x6D, 0xB6,
		0xDF, 0x3E, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5,
		0xAD, 0x00, 0x00, 0x00, 0x01, 0xB6, 0xDB, 0x6D, 0xB6, 0xDB, 0x6D, 0xBC, 0x78, 0xF1, 0xDD, 0xDD,
		0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDE, 0x3C, 0x78, 0xDB, 0x6D,
		0xBE, 0x7C, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0x00, 0x00, 0x00, 0x03, 0x6D, 0xB6, 0xDB, 0x6D, 0xB6,
		0xD8, 0x3C, 0x78, 0xF1, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
		0xDD, 0xDE, 0x3C, 0x78, 0xDB, 0x60, 0xDF, 0x3E, 0x6B, 0x5A, 0xD6, 0xD6, 0xB5, 0xA0, 0x00, 0x00,
		0x00, 0x34, 0x36, 0xDB, 0x6C, 0x1B, 0x6D, 0xB6, 0xDE, 0x3C, 0x00, 0x07, 0x77, 0x77, 0x77, 0x77,
		0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x78, 0xF0, 0x0F, 0x1B, 0x6D, 0xB7, 0xCF,
		0x9A, 0xD6, 0xB5, 0xAD, 0x6B, 0x40, 0x00, 0x00, 0x00, 0x6D, 0xB6, 0xDB, 0x03, 0x6D, 0xB6, 0xDB,
		0xC7, 0x80, 0x83, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
		0xBC, 0x78, 0x03, 0xC6, 0xDB, 0x6D, 0xF3, 0xE6, 0xB5, 0xAD, 0x6B, 0x5A, 0xDB, 0xFF, 0x00, 0x00,
		0x03, 0x7C, 0xC1, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0D, 0xF3, 0x6D, 0xB6, 0xDB, 0x6D,
		0xB6, 0xDB, 0xC7, 0x8F, 0x1D, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
		0xDD, 0xDD, 0xE3, 0xC7, 0x8D, 0xB6, 0xDB, 0xE7, 0xCD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A,
		0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5, 0xA0, 0x00, 0x00, 0x00, 0x36, 0xDB, 0x6D, 0xB6, 0xDB,
		0x6D, 0xB7, 0x8F, 0x1E, 0x3B, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
		0xBB, 0xBB, 0xC7, 0x8F, 0x1B, 0x6D, 0xB7, 0xCF, 0x9A, 0xD6, 0xB5, 0xAD, 0x6B, 0x40, 0x00, 0x00,
		0x00, 0x6D, 0xB6, 0xDB, 0x6D, 0xB6, 0xDB, 0x77, 0x8F, 0x1E, 0x3B, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
		0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xC7, 0x8F, 0x1B, 0x6D, 0xDB, 0xE7, 0xCD, 0x6B,
		0x5A, 0xD6, 0xD6, 0xB4, 0x00, 0x00, 0x00, 0x06, 0x06, 0xDB, 0x6D, 0xBB, 0x6D, 0xB6, 0xDB, 0xC7,
		0x80, 0x20, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEF,
		0x1E, 0x01, 0xE3, 0x6D, 0xB6, 0xF9, 0xF3, 0x5A, 0xD6, 0xB5, 0xAD, 0x60, 0x00, 0x00, 0x00, 0x0D,
		0xB6, 0xDB, 0x60, 0x6D, 0xB6, 0xDB, 0x78, 0xF2, 0x00, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
		0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x8F, 0x10, 0x78, 0xDB, 0x6D, 0xBE, 0x7C, 0xD6, 0xB5,
		0xAD, 0x6B, 0x5A, 0xFF, 0xE0, 0x00, 0x00, 0x6F, 0x98, 0x3E, 0x00, 0x00, 0x01, 0x06, 0x80, 0x20,
		0x47, 0x5B, 0x1C, 0x7D, 0xF2, 0xBE, 0xEE, 0x1C, 0x19, 0x1E, 0x26, 0x80, 0x00, 0x00, 0x00, 0xDF,
		0x36, 0xDB, 0x6D, 0xB6, 0xDB, 0x6D, 0xBC, 0x78, 0xF1, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
		0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDE, 0x3C, 0x78, 0xDB, 0x6D, 0xBE, 0x7C, 0xD6, 0xB5, 0xAD,
		0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAD, 0x6B, 0x5A, 0x00, 0x00, 0x00, 0x03,
		0x6D, 0xB6, 0xDB, 0x6D, 0xB6, 0xDB, 0x78, 0xF1, 0xE3, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
		0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBC, 0x78, 0xF1, 0xB6, 0xDB, 0x7C, 0xF9, 0xAD, 0x6B, 0x5A,
		0xD6, 0xB4, 0x00, 0x02, 0x00, 0x06, 0xDB, 0x6D, 0xB6, 0xDB, 0x6D, 0xB7, 0x78, 0xF1, 0xE3, 0xBB,
		0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBC, 0x78, 0xF1, 0xB6,
		0xDD, 0xBE, 0x7C, 0xD6, 0xB5, 0xA4, 0x6D, 0x6B, 0x40, 0x00, 0x00, 0x00, 0x60, 0x6D, 0xB6, 0xDB,
		0xB6, 0xDB, 0x6D, 0xBC, 0x78, 0x80, 0x0E, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
		0xEE, 0xEE, 0xEE, 0xEE, 0xF1, 0xE4, 0x1B, 0xB6, 0xDB, 0x6F, 0x9F, 0x35, 0xAD, 0x6B, 0x5A, 0xD6,
		0x80, 0x00, 0x00, 0x00, 0xDB, 0x6D, 0xB6, 0x86, 0xDB, 0x6D, 0xB7, 0x8F, 0x00, 0x07, 0x77, 0x77,
		0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x78, 0xF0, 0x07, 0x8D, 0xB6,
		0xDB, 0xE7, 0xCD, 0x6B, 0x5A, 0xD6, 0xB5, 0xAF, 0xFE, 0x00, 0x00, 0x06, 0xF9, 0x80, 0xDD, 0x45,
	};
/*
	unsigned char testdata_48khz_192kbps_768bpf[] = {
		0x0B, 0x77, 0xAA, 0xCF, 0x1C, 0x30, 0x43, 0x7F, 0xF0, 0x80, 0x01, 0x00, 0x00, 0x6A, 0xEF, 0x86,
		0x1F, 0xF1, 0xF3, 0xDA, 0xEF, 0xA0, 0xDA, 0xD7, 0x5D, 0x4B, 0xEA, 0x54, 0xDD, 0x29, 0x7D, 0x5D,
		0xEC, 0x27, 0xD0, 0xD2, 0xBE, 0xA4, 0xF9, 0xF3, 0xE8, 0x54, 0x92, 0xD7, 0x4C, 0x99, 0x52, 0x97,
		0xCF, 0x9C, 0xD2, 0xAD, 0x4A, 0x1A, 0x67, 0xCA, 0x9F, 0x3E, 0x74, 0xE5, 0xD4, 0x24, 0xCE, 0xA9,
		0x91, 0x54, 0xFC, 0x8B, 0xE9, 0xAF, 0xDD, 0x25, 0x7C, 0xF9, 0xFA, 0x64, 0xAF, 0xD0, 0x56, 0x87,
		0xFF, 0xCA, 0xF1, 0x9D, 0x39, 0x7C, 0xFC, 0x8C, 0xE1, 0xD3, 0x60, 0xC2, 0x7C, 0xEC, 0x8A, 0x57,
		0xDE, 0x53, 0xE0, 0x48, 0x90, 0xF6, 0x83, 0xF0, 0x08, 0x86, 0x18, 0x4F, 0x31, 0xC6, 0x21, 0x00,
		0xE3, 0x1C, 0x08, 0x47, 0x84, 0x00, 0x86, 0x30, 0x0C, 0x31, 0x47, 0x06, 0x1C, 0x00, 0x00, 0x00,
		0x00, 0x18, 0x61, 0xC1, 0x02, 0x18, 0x70, 0x01, 0x43, 0x02, 0x00, 0x08, 0x60, 0x00, 0x00, 0x07,
		0x14, 0x11, 0x87, 0x3A, 0x20, 0x9C, 0x20, 0x2E, 0x10, 0x0A, 0x0F, 0x32, 0xB2, 0x98, 0x36, 0x82,
		0xCE, 0x0C, 0x31, 0x9E, 0x43, 0xBB, 0x85, 0x72, 0x95, 0x55, 0x88, 0x2A, 0x00, 0x46, 0x2E, 0x40,
		0x5C, 0xE3, 0x5A, 0x0C, 0x47, 0x00, 0x41, 0x1A, 0x50, 0x34, 0x17, 0x18, 0xCB, 0x32, 0x88, 0x31,
		0xBE, 0xD5, 0x0C, 0xE2, 0xE0, 0x81, 0x61, 0x8D, 0x05, 0x1E, 0x71, 0xC4, 0x2D, 0xF7, 0xC1, 0xFC,
		0x94, 0xD4, 0x0E, 0x4B, 0x6A, 0x6F, 0xE2, 0xF6, 0x03, 0xFC, 0x02, 0x7D, 0x3E, 0x08, 0x09, 0xF2,
		0x94, 0x00, 0x02, 0xDC, 0x00, 0x86, 0x20, 0x80, 0x61, 0x00, 0x00, 0x10, 0xDB, 0xC8, 0x00, 0xC0,
		0x00, 0x10, 0x00, 0x0E, 0x00, 0x38, 0x20, 0x80, 0x00, 0x00, 0x06, 0x11, 0x80, 0x00, 0x01, 0x80,
		0x10, 0x00, 0x0E, 0x20, 0x80, 0x06, 0x38, 0x88, 0x23, 0x08, 0xBB, 0x77, 0x30, 0x88, 0x1B, 0x63,
		0x0D, 0x0A, 0x5C, 0x0B, 0x29, 0x55, 0xEC, 0xC9, 0x2F, 0xCE, 0x06, 0x40, 0x4B, 0x38, 0xF9, 0xE1,
		0xA2, 0x19, 0x12, 0x00, 0xA0, 0x7A, 0x18, 0x66, 0xBC, 0x05, 0x3B, 0x55, 0x38, 0x3B, 0x8D, 0xAE,
		0xB9, 0x22, 0x22, 0x0A, 0xFB, 0x20, 0x10, 0x73, 0xCA, 0x6C, 0xA8, 0xA2, 0x55, 0x39, 0xC5, 0x8F,
		0x72, 0x40, 0xC4, 0x30, 0x38, 0x7F, 0xC7, 0x4E, 0x9F, 0x3E, 0x75, 0x0D, 0x2B, 0xE8, 0x6E, 0x9E,
		0xC0, 0x19, 0x0A, 0x1A, 0x5A, 0xF3, 0x69, 0x3F, 0x56, 0x22, 0x15, 0x27, 0xCF, 0x9F, 0x2A, 0x4B,
		0x4A, 0x13, 0xF6, 0xB3, 0x9F, 0x43, 0x72, 0xF9, 0xED, 0xB7, 0x4A, 0x9F, 0x3A, 0x53, 0x4A, 0x95,
		0x27, 0xD0, 0x95, 0x3E, 0x7E, 0x39, 0x2B, 0xE8, 0x35, 0xE6, 0xBF, 0x74, 0xE6, 0xB3, 0xE7, 0xCB,
		0x6B, 0x52, 0xAE, 0xE6, 0x96, 0x1E, 0xD0, 0x60, 0x63, 0x84, 0x87, 0x08, 0x06, 0x30, 0x06, 0x00,
		0x04, 0x00, 0x80, 0x00, 0x20, 0x08, 0x38, 0xC0, 0x00, 0x80, 0x30, 0x60, 0x00, 0x04, 0x0C, 0x38,
		0x20, 0x03, 0x0C, 0x00, 0x01, 0xC0, 0x0C, 0x00, 0x00, 0x10, 0x00, 0x30, 0x8C, 0x61, 0xC2, 0x00,
		0x23, 0x84, 0x30, 0xF6, 0x02, 0x12, 0x98, 0x41, 0xE0, 0x51, 0x5D, 0xCD, 0x56, 0x39, 0xAB, 0x5D,
		0xA8, 0x97, 0x14, 0xBB, 0xCB, 0x63, 0x2C, 0x38, 0xDB, 0x38, 0x17, 0xB0, 0x0F, 0xB8, 0xD7, 0x5C,
		0x00, 0xC7, 0x22, 0xA2, 0x06, 0xD4, 0xB8, 0x30, 0x46, 0x00, 0x84, 0xE7, 0x19, 0xCC, 0x48, 0x52,
		0x83, 0x46, 0x77, 0x1B, 0x1D, 0xD3, 0xEF, 0x2C, 0x90, 0x8F, 0x86, 0x2A, 0x90, 0xBE, 0x2E, 0x60,
		0x6C, 0x98, 0xDA, 0x10, 0xDA, 0xC5, 0x84, 0x3E, 0xC0, 0x9D, 0x03, 0x02, 0x42, 0x9E, 0x13, 0x80,
		0xE0, 0x10, 0x42, 0x3C, 0x71, 0x86, 0x10, 0x80, 0x04, 0x00, 0x86, 0x00, 0x00, 0x08, 0x60, 0x00,
		0x00, 0x30, 0x60, 0x20, 0x0C, 0x1C, 0x00, 0x00, 0x03, 0x02, 0x08, 0x00, 0x00, 0x02, 0x01, 0x00,
		0x38, 0x00, 0x41, 0x0C, 0x38, 0x87, 0x00, 0x30, 0x00, 0x04, 0x5A, 0x20, 0x77, 0xDB, 0xD8, 0xF8,
		0x1B, 0xD1, 0x76, 0xE6, 0x5E, 0xDE, 0xCB, 0xD5, 0x22, 0x70, 0x8B, 0x39, 0x5A, 0x63, 0xE1, 0x39,
		0xC2, 0x7B, 0x9A, 0x0A, 0xF8, 0x01, 0x08, 0x03, 0x20, 0x29, 0x01, 0x1A, 0x01, 0x0E, 0x78, 0x2F,
		0x90, 0xA7, 0xC6, 0x30, 0x4B, 0x28, 0xB2, 0xE3, 0x09, 0x00, 0xAE, 0x9C, 0x12, 0xC7, 0x74, 0x3D,
		0xB0, 0xA9, 0x94, 0x09, 0xDC, 0x87, 0xB7, 0x6E, 0x18, 0x00, 0xF6, 0x03, 0xE0, 0x1C, 0xC0, 0x38,
		0x0F, 0x08, 0x8C, 0x10, 0x9C, 0x01, 0x08, 0x18, 0xC2, 0x80, 0x10, 0x00, 0x01, 0x80, 0x30, 0x41,
		0x04, 0x00, 0x10, 0x00, 0x80, 0x0E, 0x08, 0x00, 0x02, 0x0C, 0x08, 0x10, 0xA1, 0xC1, 0x86, 0x31,
		0x8C, 0x20, 0x02, 0x10, 0xC1, 0x00, 0x10, 0x07, 0x70, 0x01, 0x18, 0x00, 0x1F, 0xB0, 0x0F, 0x6F,
		0x7A, 0xE4, 0x18, 0xBA, 0xE9, 0x5D, 0x4A, 0x76, 0x2D, 0xC4, 0x67, 0x7A, 0x35, 0x15, 0xA8, 0x19,
		0xAC, 0x00, 0x02, 0x84, 0x09, 0x0A, 0xBB, 0xC1, 0x70, 0x25, 0x2C, 0x40, 0x96, 0x12, 0x2D, 0xF6,
		0x04, 0x81, 0x21, 0x86, 0x00, 0xB8, 0x47, 0x8C, 0xC9, 0x01, 0x84, 0x74, 0x0F, 0x82, 0x6A, 0xDA,
		0x42, 0x71, 0x70, 0x4D, 0x77, 0x3A, 0xC5, 0x07, 0x76, 0xFB, 0x8E, 0xE3, 0xF6, 0x84, 0xE9, 0x8C,
		0xE0, 0x04, 0xF3, 0x14, 0xC0, 0x38, 0x0C, 0x70, 0x00, 0x84, 0x31, 0x8E, 0x08, 0xE1, 0x00, 0x38,
		0x40, 0x08, 0x80, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x40, 0x80, 0x00, 0x08, 0x00, 0x01,
	};
*/
	public ref class Test_Base
	{
	protected:
		AC3SOURCE* m_source;
		const unsigned char* m_data;
		int m_size;

		DWORD m_expectedFrameSize;
		DWORD m_expectedChannelCount;
		DWORD m_expectedFrequency;
		DWORD m_expectedBytesPerSecond;
		__int64 m_expectedFrameDuration;

	public:
		Test_Base(const unsigned char* source, int size,
			DWORD expectedFrameSize, DWORD expectedChannels, 
			__int64 expectedFrameDuration, DWORD expectedFrequency,
			DWORD expectedBytesPerSecond)
		{
			m_data = source;
			m_size = size;
			m_expectedFrameSize = expectedFrameSize;
			m_expectedChannelCount = expectedChannels;
			m_expectedFrameDuration = expectedFrameDuration;
			m_expectedFrequency = expectedFrequency;
			m_expectedBytesPerSecond = expectedBytesPerSecond;
		}

		[SetUp]
		void Setup()
		{
			m_source = GetAc3AudioSourceFromBuffer(m_data, m_size);
		}

		[TearDown]
		void Teardown()
		{
			CloseAc3AudioSource(m_source);
		}

		[Test]
		void Test_FrameSize()
		{
			DWORD frameSize = m_source->GetFrameSize();
			Assert::AreEqual((unsigned __int64)m_expectedFrameSize, (unsigned __int64)frameSize);
		}

		[Test]
		void Test_ChannelCount()
		{
			DWORD channels = m_source->GetChannelCount();
			Assert::AreEqual((unsigned __int64)m_expectedChannelCount, (unsigned __int64)channels);
		}

		[Test]
		void Test_FrameDuration()
		{
			__int64 duration = m_source->GetFrameDuration();
			Assert::AreEqual((__int64)m_expectedFrameDuration, (__int64)duration);
		}

		[Test]
		void Test_Frequency()
		{
			__int64 frequency = m_source->GetFrequency();
			Assert::AreEqual((__int64)m_expectedFrequency, (__int64)frequency);
		}

		[Test]
		void Test_Bitrate()
		{
			__int64 bytesPerSec = m_source->GetAvgBytesPerSec();
			Assert::AreEqual((__int64)m_expectedBytesPerSecond, (__int64)bytesPerSec);
		}

		[Test]
		void Test_ReadFrame_old()
		{
			std::vector<unsigned char> buffer;
			buffer.resize(m_expectedFrameSize);

			DWORD dwMicroSecRead;
			__int64 nanoSecRead;
			__int64 timeCode;
			DWORD bytesRead = m_source->Read(&buffer[0], 1, &dwMicroSecRead, &nanoSecRead, &timeCode);

			Assert::AreEqual((unsigned __int64)bytesRead, (unsigned __int64)m_expectedFrameSize);
			Assert::AreEqual((unsigned __int64)dwMicroSecRead, (unsigned __int64)(m_expectedFrameDuration / 1000));
			Assert::AreEqual((unsigned __int64)nanoSecRead, (unsigned __int64)m_expectedFrameDuration);
			Assert::AreEqual((unsigned __int64)timeCode, (unsigned __int64)0);
		}

		[Test]
		void Test_ReadFrame_new()
		{
			MULTIMEDIA_DATA_PACKET* packet = NULL;
			m_source->SetFrameMode(1);
			DWORD dwRead = m_source->Read(&packet);

			Assert::AreEqual((unsigned __int64)packet->totalDataSize, (unsigned __int64)m_expectedFrameSize);
			Assert::AreEqual((unsigned __int64)packet->duration, (unsigned __int64)m_expectedFrameDuration);
			
			// nextTimecode muss -1 sein, weil der Stream hier nach der ersten Frame zu ende ist
			Assert::AreEqual((__int64)packet->nextTimecode, (__int64)-1);
			Assert::AreEqual((unsigned __int64)packet->timecode, (unsigned __int64)0);

			freeMultimediaDataPacket(packet);
		}
	};

	[TestFixture]
	public ref class Test_448kbps_1792kbps_6ch : Test_Base
	{
	public:
		Test_448kbps_1792kbps_6ch()	: 
		  Test_Base(testdata_48khz_448kbps_1792bpf, sizeof(testdata_48khz_448kbps_1792bpf),
			  1792, 6, 32000000, 48000, 448000 >> 3)
		{
		}
	};

	[TestFixture]
	public ref class Test_384kbps_1536kbps_6ch : Test_Base
	{
	public:
		Test_384kbps_1536kbps_6ch()	: 
		  Test_Base(testdata_48khz_384kbps_1536bpf, sizeof(testdata_48khz_384kbps_1536bpf),
			  1536, 6, 32000000, 48000, 384000 >> 3)
		{
		}
	};

/*	[TestFixture]
	public ref class Test_192kbps_768kbps_2ch : Test_Base
	{
	public:
		Test_192kbps_768kbps_2ch()	: 
		  Test_Base(testdata_48khz_192kbps_768bpf, sizeof(testdata_48khz_192kbps_768bpf),
			  768, 2, 32000000, 48000, 192000 >> 3)
		{
		}
	};*/
}}